/**
 *   This file is part of Skript.
 *
 *  Skript is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Skript is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Skript.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright Peter Güttinger, SkriptLang team and contributors
 */
package ch.njol.skript;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.bukkit.Bukkit;
import org.bukkit.command.Command;
import org.bukkit.command.CommandExecutor;
import org.bukkit.command.CommandSender;
import org.bukkit.plugin.PluginDescriptionFile;
import org.eclipse.jdt.annotation.Nullable;

import ch.njol.skript.aliases.Aliases;
import ch.njol.skript.command.CommandHelp;
import ch.njol.skript.config.Config;
import ch.njol.skript.doc.HTMLGenerator;
import ch.njol.skript.localization.ArgsMessage;
import ch.njol.skript.localization.Language;
import ch.njol.skript.localization.PluralizingArgsMessage;
import ch.njol.skript.log.RedirectingLogHandler;
import ch.njol.skript.tests.runner.SkriptTestEvent;
import ch.njol.skript.tests.runner.TestMode;
import ch.njol.skript.tests.runner.TestTracker;
import ch.njol.skript.util.ExceptionUtils;
import ch.njol.skript.util.FileUtils;
import ch.njol.skript.util.SkriptColor;
import ch.njol.util.StringUtils;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

public class SkriptCommand implements CommandExecutor {
	
	private static final String CONFIG_NODE = "skript command";
	
	// TODO /skript scripts show/list - lists all enabled and/or disabled scripts in the scripts folder and/or subfolders (maybe add a pattern [using * and **])
	// TODO document this command on the website
	private static final CommandHelp skriptCommandHelp = new CommandHelp("<gray>/<gold>skript", SkriptColor.LIGHT_CYAN, CONFIG_NODE + ".help")
		.add(new CommandHelp("reload", SkriptColor.DARK_RED)
			.add("all")
			.add("config")
			.add("aliases")
			.add("scripts")
			.add("<script>")
		).add(new CommandHelp("enable", SkriptColor.DARK_RED)
			.add("all")
			.add("<script>")
		).add(new CommandHelp("disable", SkriptColor.DARK_RED)
			.add("all")
			.add("<script>")
		).add(new CommandHelp("update", SkriptColor.DARK_RED)
			.add("check")
			.add("changes")
			.add("download")
		).add("info"
		).add("help");
	
	static {
		if (new File(Skript.getInstance().getDataFolder() + "/doc-templates").exists()) {
			skriptCommandHelp.add("gen-docs");
		}
		if (TestMode.DEV_MODE) { // Add command to run individual tests
			skriptCommandHelp.add("test");
		}
	}
	
	private static final ArgsMessage m_reloading = new ArgsMessage(CONFIG_NODE + ".reload.reloading");
	
	private static void reloading(CommandSender sender, String what, Object... args) {
		what = args.length == 0 ? Language.get(CONFIG_NODE + ".reload." + what) : Language.format(CONFIG_NODE + ".reload." + what, args);
		Skript.info(sender, StringUtils.fixCapitalization(m_reloading.toString(what)));
	}
	
	private static final ArgsMessage m_reloaded = new ArgsMessage(CONFIG_NODE + ".reload.reloaded");
	private static final ArgsMessage m_reload_error = new ArgsMessage(CONFIG_NODE + ".reload.error");
	
	private static void reloaded(CommandSender sender, RedirectingLogHandler r, String what, Object... args) {
		what = args.length == 0 ? Language.get(CONFIG_NODE + ".reload." + what) : PluralizingArgsMessage.format(Language.format(CONFIG_NODE + ".reload." + what, args));
		if (r.numErrors() == 0)
			Skript.info(sender, StringUtils.fixCapitalization(PluralizingArgsMessage.format(m_reloaded.toString(what))));
		else
			Skript.error(sender, StringUtils.fixCapitalization(PluralizingArgsMessage.format(m_reload_error.toString(what, r.numErrors()))));
	}
	
	private static void info(CommandSender sender, String what, Object... args) {
		what = args.length == 0 ? Language.get(CONFIG_NODE + "." + what) : PluralizingArgsMessage.format(Language.format(CONFIG_NODE + "." + what, args));
		Skript.info(sender, StringUtils.fixCapitalization(what));
	}
	
	private static void error(CommandSender sender, String what, Object... args) {
		what = args.length == 0 ? Language.get(CONFIG_NODE + "." + what) : PluralizingArgsMessage.format(Language.format(CONFIG_NODE + "." + what, args));
		Skript.error(sender, StringUtils.fixCapitalization(what));
	}
	
	@Override
	@SuppressFBWarnings("REC_CATCH_EXCEPTION")
	public boolean onCommand(@Nullable CommandSender sender, @Nullable Command command, @Nullable String label, @Nullable String[] args) {
		if (sender == null || command == null || label == null || args == null)
			throw new IllegalArgumentException();
		if (!skriptCommandHelp.test(sender, args))
			return true;
		try (RedirectingLogHandler logHandler = new RedirectingLogHandler(sender, "").start()) {
			if (args[0].equalsIgnoreCase("reload")) {
				if (args[1].equalsIgnoreCase("all")) {
					reloading(sender, "config, aliases and scripts");
					SkriptConfig.load();
					Aliases.clear();
					Aliases.load();
					
					if (!ScriptLoader.isAsync())
						ScriptLoader.disableScripts();
					
					ScriptLoader.loadScripts(logHandler)
						.thenAccept(unused ->
							reloaded(sender, logHandler, "config, aliases and scripts"));
				} else if (args[1].equalsIgnoreCase("scripts")) {
					reloading(sender, "scripts");
					
					if (!ScriptLoader.isAsync())
						ScriptLoader.disableScripts();
					
					ScriptLoader.loadScripts(logHandler)
						.thenAccept(unused ->
							reloaded(sender, logHandler, "scripts"));
				} else if (args[1].equalsIgnoreCase("config")) {
					reloading(sender, "main config");
					SkriptConfig.load();
					reloaded(sender, logHandler, "main config");
				} else if (args[1].equalsIgnoreCase("aliases")) {
					reloading(sender, "aliases");
					Aliases.clear();
					Aliases.load();
					reloaded(sender, logHandler, "aliases");
				} else {
					File f = getScriptFromArgs(sender, args, 1);
					if (f == null)
						return true;
					if (!f.isDirectory()) {
						if (f.getName().startsWith("-")) {
							info(sender, "reload.script disabled", f.getName().substring(1), StringUtils.join(args, " ", 1, args.length));
							return true;
						}
						reloading(sender, "script", f.getName());
						ScriptLoader.reloadScript(f, logHandler)
							.thenAccept(scriptInfo ->
								reloaded(sender, logHandler, "script", f.getName()));
					} else {
						reloading(sender, "scripts in folder", f.getName());
						ScriptLoader.reloadScripts(f, logHandler)
							.thenAccept(scriptInfo -> {
								int enabled = scriptInfo.files;
								if (enabled == 0)
									info(sender, "reload.empty folder", f.getName());
								else
									reloaded(sender, logHandler, "x scripts in folder", f.getName(), enabled);
							});
					}
				}
			} else if (args[0].equalsIgnoreCase("enable")) {
				if (args[1].equals("all")) {
					try {
						info(sender, "enable.all.enabling");
						File[] files = toggleScripts(new File(Skript.getInstance().getDataFolder(), Skript.SCRIPTSFOLDER), true).toArray(new File[0]);
						List<Config> configs = ScriptLoader.loadStructures(files);
						ScriptLoader.loadScripts(configs, logHandler)
							.thenAccept(scriptInfo -> {
								if (logHandler.numErrors() == 0) {
									info(sender, "enable.all.enabled");
								} else {
									error(sender, "enable.all.error", logHandler.numErrors());
								}
							});
					} catch (IOException e) {
						error(sender, "enable.all.io error", ExceptionUtils.toString(e));
					}
				} else {
					File f = getScriptFromArgs(sender, args, 1);
					if (f == null)
						return true;
					if (!f.isDirectory()) {
						if (!f.getName().startsWith("-")) {
							info(sender, "enable.single.already enabled", f.getName(), StringUtils.join(args, " ", 1, args.length));
							return true;
						}
						
						try {
							f = FileUtils.move(f, new File(f.getParentFile(), f.getName().substring(1)), false);
						} catch (IOException e) {
							error(sender, "enable.single.io error", f.getName().substring(1), ExceptionUtils.toString(e));
							return true;
						}
						
						info(sender, "enable.single.enabling", f.getName());
						Config config = ScriptLoader.loadStructure(f);
						if (config == null)
							return true;
						
						String fileName = f.getName();
						ScriptLoader.loadScripts(Collections.singletonList(config), logHandler)
							.thenAccept(scriptInfo -> {
								if (logHandler.numErrors() == 0) {
									info(sender, "enable.single.enabled", fileName);
								} else {
									error(sender, "enable.single.error", fileName, logHandler.numErrors());
								}
							});
						return true;
					} else {
						Collection<File> scripts;
						try {
							scripts = toggleScripts(f, true);
						} catch (IOException e) {
							error(sender, "enable.folder.io error", f.getName(), ExceptionUtils.toString(e));
							return true;
						}
						if (scripts.isEmpty()) {
							info(sender, "enable.folder.empty", f.getName());
							return true;
						}
						info(sender, "enable.folder.enabling", f.getName(), scripts.size());
						File[] ss = scripts.toArray(new File[0]);
						
						List<Config> configs = ScriptLoader.loadStructures(ss);
						if (configs.size() == 0)
							return true;
						
						String fileName = f.getName();
						ScriptLoader.loadScripts(configs, logHandler)
							.thenAccept(scriptInfo -> {
								if (logHandler.numErrors() == 0) {
									info(sender, "enable.folder.enabled", fileName, scriptInfo.files);
								} else {
									error(sender, "enable.folder.error", fileName, logHandler.numErrors());
								}
							});
						return true;
					}
				}
			} else if (args[0].equalsIgnoreCase("disable")) {
				if (args[1].equals("all")) {
					ScriptLoader.disableScripts();
					try {
						toggleScripts(new File(Skript.getInstance().getDataFolder(), Skript.SCRIPTSFOLDER), false);
						info(sender, "disable.all.disabled");
					} catch (IOException e) {
						error(sender, "disable.all.io error", ExceptionUtils.toString(e));
					}
				} else {
					File f = getScriptFromArgs(sender, args, 1);
					if (f == null) // TODO allow disabling deleted/renamed scripts
						return true;
					
					if (!f.isDirectory()) {
						if (f.getName().startsWith("-")) {
							info(sender, "disable.single.already disabled", f.getName().substring(1));
							return true;
						}
						
						ScriptLoader.unloadScript(f);
						
						try {
							FileUtils.move(f, new File(f.getParentFile(), "-" + f.getName()), false);
						} catch (IOException e) {
							error(sender, "disable.single.io error", f.getName(), ExceptionUtils.toString(e));
							return true;
						}
						info(sender, "disable.single.disabled", f.getName());
					} else {
						Collection<File> scripts;
						try {
							scripts = toggleScripts(f, false);
						} catch (IOException e) {
							error(sender, "disable.folder.io error", f.getName(), ExceptionUtils.toString(e));
							return true;
						}
						if (scripts.isEmpty()) {
							info(sender, "disable.folder.empty", f.getName());
							return true;
						}
						
						for (File script : scripts)
							ScriptLoader.unloadScript(new File(script.getParentFile(), script.getName().substring(1)));
						
						info(sender, "disable.folder.disabled", f.getName(), scripts.size());
					}
					return true;
				}
			} else if (args[0].equalsIgnoreCase("update")) {
				SkriptUpdater updater = Skript.getInstance().getUpdater();
				if (updater == null) { // Oh. That is bad
					Skript.info(sender, "" + SkriptUpdater.m_internal_error);
					return true;
				}
				if (args[1].equals("check")) {
					updater.updateCheck(sender);
				} else if (args[1].equalsIgnoreCase("changes")) {
					updater.changesCheck(sender);
				} else if (args[1].equalsIgnoreCase("download")) {
					updater.updateCheck(sender);
				}
			} else if (args[0].equalsIgnoreCase("info")) {
				info(sender, "info.aliases");
				info(sender, "info.documentation");
				info(sender, "info.server", Bukkit.getVersion());
				info(sender, "info.version", Skript.getVersion());
				info(sender, "info.addons");
				for (SkriptAddon addon : Skript.getAddons()) {
					PluginDescriptionFile desc = addon.plugin.getDescription();
					String web = desc.getWebsite();
					Skript.info(sender, " - " + desc.getFullName() + (web != null ? " (" + web + ")" : ""));
				}
			} else if (args[0].equalsIgnoreCase("help")) {
				skriptCommandHelp.showHelp(sender);
			} else if (args[0].equalsIgnoreCase("gen-docs")) {
				File templateDir = new File(Skript.getInstance().getDataFolder() + "/doc-templates/");
				if (!templateDir.exists()) {
					Skript.info(sender, "Documentation templates not found. Cannot generate docs!");
					return true;
				}
				File outputDir = new File(Skript.getInstance().getDataFolder() + "/docs");
				outputDir.mkdirs();
				HTMLGenerator generator = new HTMLGenerator(templateDir, outputDir);
				Skript.info(sender, "Generating docs...");
				generator.generate(); // Try to generate docs... hopefully
				Skript.info(sender, "Documentation generated!");
			} else if (args[0].equalsIgnoreCase("test") && TestMode.DEV_MODE) {
				File script;
				if (args.length == 1) {
					script = TestMode.lastTestFile;
					if (script == null) {
						Skript.error(sender, "No test script has been run yet!");
						return true;
					}
				} else {
					script = TestMode.TEST_DIR.resolve(
							Arrays.stream(args).skip(1).collect(Collectors.joining(" ")) + ".sk").toFile();
					TestMode.lastTestFile = script;
				}
				if (!script.exists()) {
					Skript.error(sender, "Test script doesn't exist!");
					return true;
				}
				
				Config config = ScriptLoader.loadStructure(script);
				if (config != null) {
					ScriptLoader.loadScripts(Collections.singletonList(config), logHandler)
						.thenAccept(scriptInfo ->
							// Code should run on server thread
							Bukkit.getScheduler().scheduleSyncDelayedTask(Skript.getInstance(), () -> {
								Bukkit.getPluginManager().callEvent(new SkriptTestEvent()); // Run it
								// ScriptLoader.disableScripts(); // Clean state for next test
								
								// Get results and show them
								String[] lines = TestTracker.collectResults().createReport().split("\n");
								for (String line : lines) {
									Skript.info(sender, line);
								}
							}));
				}
			}
		} catch (Exception e) {
			Skript.exception(e, "Exception occurred in Skript's main command", "Used command: /" + label + " " + StringUtils.join(args, " "));
		}
		return true;
	}
	
	private static final ArgsMessage m_invalid_script = new ArgsMessage(CONFIG_NODE + ".invalid script");
	private static final ArgsMessage m_invalid_folder = new ArgsMessage(CONFIG_NODE + ".invalid folder");
	
	@Nullable
	private static File getScriptFromArgs(CommandSender sender, String[] args, int start) {
		String script = StringUtils.join(args, " ", start, args.length);
		File f = getScriptFromName(script);
		if (f == null){
			Skript.error(sender, (script.endsWith("/") || script.endsWith("\\") ? m_invalid_folder : m_invalid_script).toString(script));
			return null;
		}
		return f;
	}
	
	@Nullable
	public static File getScriptFromName(String script){
		boolean isFolder = script.endsWith("/") || script.endsWith("\\");
		if (isFolder) {
			script = script.replace('/', File.separatorChar).replace('\\', File.separatorChar);
		} else if (!StringUtils.endsWithIgnoreCase(script, ".sk")) {
			int dot = script.lastIndexOf('.');
			if (dot > 0 && !script.substring(dot+1).equals("")) {
				return null;
			}
			script = script + ".sk";
		}
		if (script.startsWith("-"))
			script = script.substring(1);
		File f = new File(Skript.getInstance().getDataFolder(), Skript.SCRIPTSFOLDER + File.separator + script);
		if (!f.exists()) {
			f = new File(f.getParentFile(), "-" + f.getName());
			if (!f.exists()) {
				return null;
			}
		}
		return f;
	}
	
	private static Collection<File> toggleScripts(File folder, boolean enable) throws IOException {
		return FileUtils.renameAll(folder, name -> {
			if (StringUtils.endsWithIgnoreCase(name, ".sk") && name.startsWith("-") == enable)
				return enable ? name.substring(1) : "-" + name;
			return null;
		});
	}
	
}
