/**
 *   This file is part of Skript.
 *
 *  Skript is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Skript is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Skript.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright Peter Güttinger, SkriptLang team and contributors
 */
package ch.njol.skript.lang.function;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.eclipse.jdt.annotation.Nullable;

/**
 * Contains a set of functions.
 */
public class Namespace {
	
	/**
	 * Origin of functions in namespace.
	 */
	public enum Origin {
		/**
		 * Functions implemented in Java.
		 */
		JAVA,
		
		/**
		 * Script functions.
		 */
		SCRIPT
	}
	
	/**
	 * Key to a namespace.
	 */
	public static class Key {
		
		private final Origin origin;

		@Nullable
		private final String scriptName;

		public Key(Origin origin, @Nullable String scriptName) {
			super();
			this.origin = origin;
			this.scriptName = scriptName;
		}
		
		public Origin getOrigin() {
			return origin;
		}

		@Nullable
		public String getScriptName() {
			return scriptName;
		}

		@Override
		public int hashCode() {
			int result = origin.hashCode();
			result = 31 * result + (scriptName != null ? scriptName.hashCode() : 0);
			return result;
		}

		@Override
		public boolean equals(Object object) {
			if (this == object)
				return true;
			if (object == null || getClass() != object.getClass())
				return false;

			Key other = (Key) object;

			if (origin != other.origin)
				return false;
			return Objects.equals(scriptName, other.scriptName);
		}
	}

	/**
	 * The key used in the signature and function maps
	 */
	private static class Info {

		/**
		 * Name of the function
		 */
		private final String name;

		/**
		 * Whether the function is local
		 */
		private final boolean local;

		public Info(String name, boolean local) {
			this.name = name;
			this.local = local;
		}

		public String getName() {
			return name;
		}

		public boolean isLocal() {
			return local;
		}

		@Override
		public int hashCode() {
			int result = getName().hashCode();
			result = 31 * result + (isLocal() ? 1 : 0);
			return result;
		}

		@Override
		public boolean equals(Object o) {
			if (this == o)
				return true;
			if (!(o instanceof Info))
				return false;

			Info info = (Info) o;

			if (isLocal() != info.isLocal())
				return false;
			return getName().equals(info.getName());
		}
	}
	
	/**
	 * Signatures of known functions.
	 */
	private final Map<Info, Signature<?>> signatures;

	/**
	 * Known functions. Populated as function bodies are loaded.
	 */
	private final Map<Info, Function<?>> functions;

	public Namespace() {
		this.signatures = new HashMap<>();
		this.functions = new HashMap<>();
	}
	
	@Nullable
	public Signature<?> getSignature(String name, boolean local) {
		return signatures.get(new Info(name, local));
	}

	@Nullable
	public Signature<?> getSignature(String name) {
		Signature<?> signature = getSignature(name, true);
		return signature == null ? getSignature(name, false) : signature;
	}

	public void addSignature(Signature<?> sign) {
		Info info = new Info(sign.getName(), sign.local);
		if (signatures.containsKey(info))
			throw new IllegalArgumentException("function name already used");
		signatures.put(info, sign);
	}

	public boolean removeSignature(Signature<?> sign) {
		Info info = new Info(sign.getName(), sign.local);
		if (signatures.get(info) != sign)
			return false;
		signatures.remove(info);
		return true;
	}
	
	@SuppressWarnings("null")
	public Collection<Signature<?>> getSignatures() {
		return signatures.values();
	}
	
	@Nullable
	public Function<?> getFunction(String name, boolean local) {
		return functions.get(new Info(name, local));
	}

	@Nullable
	public Function<?> getFunction(String name) {
		Function<?> function = getFunction(name, true);
		return function == null ? getFunction(name, false) : function;
	}

	public void addFunction(Function<?> func) {
		Info info = new Info(func.getName(), func.getSignature().local);
		assert signatures.containsKey(info) : "missing signature for function";
		functions.put(info, func);
	}

	@SuppressWarnings("null")
	public Collection<Function<?>> getFunctions() {
		return functions.values();
	}
}
